function F = tar_grw(alp,fun,m0,s0)
%--------------------------------------------------------------------------
% tar_grw.m         Log of the posterior distribution of alpha and log phi.
%                   Data subject to Gaussian noise.
%                   Alpha has non-informative prior and Phi has gamma prior
%                   characterised by a degree of belief term, m0 and
%                   standard deviation term s0.
%                   $\phi \sim \rm{G}(m_0/2,m_0 s_0^2/2)$
%--------------------------------------------------------------------------
% K Jagan, A B Forbes, NPL, www.npl.co.uk
%--------------------------------------------------------------------------
% Inputs:
% alp(p,N)          Matrix of alphas and log phi for N parallel chains.
% fun               Function handle that takes as input alpha values 
%                   and returns the non linear least squares function, f,
%                   and the matrix of derivatives of f with respect to each
%                   input, J. 
% m0                Prior degrees of freedom.
% s0                Prior estimate of standard deviation. 
% Output:
% F(1,N)            Log of the posterior distribution of alpha and log phi.
%--------------------------------------------------------------------------
% F = tar_grw(alp,fun,m0,s0)
%--------------------------------------------------------------------------
% Number of model parameters 
  p = size(alp,1);
  n = p-1;
 
% Extract alphas and phis from input matrix 
  A = alp(1:n,:);
  P = exp(alp(n+1,:));

% Evaluating the function and determining the length of the data vector
  f = fun(A);
  m = size(f,1);

% Log likelihood
  lik = (m/2)*log(P) - (P/2).*sum(f.^2);

% Log of the prior
  al = m0/2;
  be = (m0*s0^2)/2;
  pri = (al)*log(P) - be*P;

% Log of the posterior
  F = lik + pri;
%--------------------------------------------------------------------------
% End of tar_grw.m
%--------------------------------------------------------------------------
end