function [abar, s, aQ] = mcsums(A, M0, Q)
% -------------------------------------------------------------------------
% mcsums.m        Summary information from MC samples.
% -------------------------------------------------------------------------
% K Jagan, A B Forbes, NPL, www.npl.co.uk
% -------------------------------------------------------------------------
% Input parameters
% A(M,N)           An array that stores samples of size M x N.
% M0               Burn-in period with M > M0 >= 0.
% Q(nQ,1)          Percentiles specifications, 0 <= Q(l) <= 100.
%
% Output parameters
% abar(n,1)        Mean for each sample.
% s(n,1)           Standard deviation for sample.
% aQ(nQ,n)         Percentiles corresponding to Q.
%                  If Q(l) =   0, aQ(l,j) = min(A{j});
%                     Q(l) = 100, aQ(l,j) = max(A{j}).
%            Note: abar, s, and aQ are calculated from
%                  reshape(A(M0+1:M,:),(M-M0)*N,1).
% -------------------------------------------------------------------------
% [abar, s, aQ] = mcsums(A, M0, Q)
% -------------------------------------------------------------------------

% size of samples after burn-in
  [M, N] = size(A);
  m = (M - M0)*N;

  if nargout > 2
      nQ = length(Q);
      aQ = zeros(nQ, 1);
  end

% samples from N parallel chains after burn-in period
  aaj = A(M0+1:M, :);
  aaj = aaj(:);

% mean and standard deviation of samples
  abar = mean(aaj);
  s = std(aaj);

% percentiles of samples
  if nargout > 2
      aajs = sort(aaj);
      for q = 1:nQ
          iq = round(Q(q)*m/100);
          iq = max([iq; 1]);
          aQ(q) = aajs(iq);
      end
  end
% -------------------------------------------------------------------------
% End of mcsums.m
% -------------------------------------------------------------------------
end