  function [IS,IA] = mhicI(P,P0)
% -------------------------------------------------------------------------
% MHICI.M          MCMC Metropolis-Hastings independence chain (MHIC).
%                  Calculation of chain indices.
% -------------------------------------------------------------------------
% A B Forbes, K Jagan, I M Smith, NPL, www.npl.co.uk
% -------------------------------------------------------------------------
% Design documentation 
% [1] A B Forbes, An MCMC algorithm based on GUM Supplement 1 for 
% uncertainty evaluation, Measurement, 2012, 
% DOI:10.1016/j.measurement.2012.01.018
%
% [2] A B Forbes et al, User manual for MCM2MCMC: Software implementing a
% Markov chain Monte Carlo algorithm for uncertainty evaluation based on 
% Supplement 1 to the GUM, NPL Report MS/20, 2018
% -------------------------------------------------------------------------
% Input parameters
% P(M,N)           Target distribution  p(a0qr)) evaluated at a0qr. 
%                  Note: P(1,r) > 0.
% P0(M,N)          Proposal distribution p0(a0qr) evaluated at a0qr.
%                  Note: P0(q,r) > 0.
%
% Output parameters
% IS(M,N)          Selection indices.
%                  If A0(:,:) contains samples from the proposal, then
%                  A(i,j) = A0(IS(i,j),j) stores the constructed
%                  MHIC chains.
% IA(M,N)          Acceptance indices. IA(q,r) = 1, if a0qr is accepted,
%                                                0, otherwise.
% -------------------------------------------------------------------------
% [IS,IA] = mhicI(P,P0)
% -------------------------------------------------------------------------
  [M,N] = size(P);
%
% Check P and P0. 
  p0min = min(P0(:));
  if p0min <=0
      error('P0(q,r) must be strictly positive for all q and r')
  end % if p0min 
  pmin = min(P(1,:));
  if pmin <=0
      error('P(1,r) must be strictly positive for all r')
  end % if pmin 
%
% Initialisation. 
  IA = zeros(M,N);
  IA(1,:) = ones(1,N);
  IS = zeros(M,N);
  IS(1,:) = ones(1,N);
%
  p0q = P0(1,:);
  pq =  P(1,:);
%
% Run the chains. 
  for q = 2:M
%     Proposals.
      ps = P(q,:);
      p0s = P0(q,:);
%    
%     Acceptance ratios.
      rq = ps.*p0q./(pq.*p0s);
%    
%     Draws from uniform distribution.
      uq = rand(1,N);
%    
%     Acceptance and selection indices.
      Iq = uq < rq;
      IA(q,:) = Iq;
      IS(q, Iq) = q*ones(1,sum(Iq));
      IS(q,~Iq) = IS(q-1,~Iq);
%    
%     Update density values associated with current parameters.
      pq(Iq) = ps(Iq);
      p0q(Iq) = p0s(Iq);

      
  end % for q 
% -------------------------------------------------------------------------
% End of mhicI.m
% -------------------------------------------------------------------------