function T = tar_ngic(ap,fun,m0,s0)
%--------------------------------------------------------------------------
% tar_ngic.m        Log of the posterior distribution of alpha and phi.
%                   Data subject to Gaussian noise.
%                   Alpha has non-informative prior and Phi has gamma prior
%                   characterised by a degree of belief term, m0 and
%                   standard deviation term s0.
%                   $\phi \sim \rm{G}(m_0/2,m_0 s_0^2/2)$
%--------------------------------------------------------------------------
% K Jagan, A B Forbes, NPL, www.npl.co.uk
%--------------------------------------------------------------------------
% Inputs:
% ap(p,N)           Matrix of alphas and phi for N parallel chains.
% fun               Function handle that takes as input alpha values 
%                   and returns the non linear least squares function, f,
%                   and the matrix of derivatives of f with respect to each
%                   input, J. 
% m0                Prior degrees of freedom.
% s0                Prior estimate of standard deviation. 
% Output:
% T(1,N)            Log of the posterior distribution of alpha and phi.
%--------------------------------------------------------------------------
% T = tar_ngic(ap,fun,m0,s0)
%--------------------------------------------------------------------------

% Extract alphas and phis from ap 
  [n,~] = size(ap);
  A = ap(1:n-1,:);
  P = ap(n,:);

% Log likelihood 
  f = fun(A);
  m = size(f,1);
  lik = (m/2)*log(P)-(P/2).*sum(f.^2);

% Log prior
  al = m0/2;
  be = (m0*s0^2)/2;
  pri = (al-1)*log(P)-be*P;

% Log of posterior 
  T = lik+pri;
end 
%--------------------------------------------------------------------------
% End of tar_ngic.m
%--------------------------------------------------------------------------