function [F,J] = ED(a,x,x0,y,si)
%--------------------------------------------------------------------------
% ED.m              Function and Jacobian associated with exponential 
%                   decay model. 
%                   $f_i = (y_i - a_1\exp(a_2(x_i-x_0)) - a_3)/s_i$
%                   $J(i,p)$ is the derivative of $f_i$ with respect 
%                   to $a_p$.
%--------------------------------------------------------------------------
% K Jagan, A B Forbes, NPL, www.npl.co.uk
%--------------------------------------------------------------------------
% Inputs:
% a(3,1)            Parameters of the exponential decay model. 
% x(m,1)            Input quantity.
% x0                Perturbation of x. 
% y(m,1)            Output data. 
% si(m,1)           Standard deviation of each y.
% Outputs:
% F(m,1)            Function values.
% J(m,3)            Gradients of F with respect to each parameter a.
%--------------------------------------------------------------------------
% [F,J] = ED(a,x,x0,y,si)                               
%--------------------------------------------------------------------------
 
% Size of data 
  m = length(x);
  [~,N] = size(a);
 
% Setting input quantities as m x N matrices to allow for chains to be 
% evaluated in parallel  
  A = repmat(a(1,:),m,1); 
  B = repmat(a(2,:),m,1);
  C = repmat(a(3,:),m,1);
  X = repmat(x,1,N);
  Y = repmat(y,1,N);
  S = repmat(si,1,N);

% Exponential decay function 
  H = C + A.*exp(-B.*(X-x0));

% Weighted F
  F = (Y-H)./S; 

% Jacobian 
  if nargout>1 
      J = zeros(m,3);
      J(:,1) = -exp(-B.*(X-x0))./S;
      J(:,2) = A.*(X-x0).*exp(-B.*(X-x0))./S;
      J(:,3) = -1*ones(m,1)./S;
  end 
% -------------------------------------------------------------------------
% End of ED.m
% -------------------------------------------------------------------------
end 