%% NPL NLLSMH Software: Release 1.1

%% Introduction
% This MATLAB software implements the Metropolis-Hastings Markov
% chain Monte Carlo (MCMC) algorithm to generate samples from a
% Bayesian posterior distribution of parameters of a non-linear model
% along with the precision parameter associated with the data.
%
% For a given measurement problem, the software performs the following
% tasks:
%
% 1. Import data.
%
% 2. Find non-linear least squares estimates of the model parameters.
%
% 3. Generate MCMC samples using 4 methods - Gaussian random walk, Normal
% Gamma independence chain, Student's t random walk, Student's t
% independence chain.
%
% 4. Extract and display key summary information for the posterior
% distribution.

%% Exponential decay example
%
% The exponential decay model takes the form
% $$ y_i = \alpha_1 \exp(\alpha_2 (x_i-x_0)) + \alpha_3 + \epsilon_i $$,
% $$ \epsilon_i \sim N(0,\phi^{-1} \sigma_i^2) $$.
%
% Non-linear least squares estimates are obtained by minimising
% $$ F = \frac{1}{2} f^{\top}f $$,
% where $f_i = (y_i - \alpha_1\exp(\alpha_2 (x_i-x_0)) - \alpha_3)/\sigma_i$.
%
% Samples are drawn from the joint posterior of $\alpha$ and $\log(\phi)$
% using the Gaussian random walk algorithm.
%
% Samples from $\alpha$ and $\phi$ are obtained using the Normal gamma
% independence chain algorithm.
%
% Samples from the marginal posterior distribution of $\alpha$ are obtained
% using the t random walk and t independence chain algorithms.

%% Licence agreement
%
% The software is provided with a software user licence agreement and the
% use of the software is subject to the terms laid out in that agreement.
% By running the software, the user accepts the terms of the agreement.

%% Main program

%%
% Identify the software to the user
  fprintf('\nNPL NLLSMH Software: Release 1.1');
  fprintf('\nNational Physical Laboratory, UK');
  fprintf('\nJanuary 2025 \n');
  fprintf('\nExponential decay example \n\n');

%%
% Close all figures and clear all variables
  close all
  clear
  format short

%% 
% Set seed for random number generation
  rng(10);
  
%%
% Assign chain length, number of chains and burn-in period
  M  = 25000;
  N  = 100;
  M0 = 5000;

%%
% Import and extract data
  EDdata

  t = TY(:,1);
  y = TY(:,2);
  m = length(t);

%%
% Fit a cubic polynomial to the data to find initial parameter estimates
  C = [ones(m,1) t t.*t t.*t.*t];
  d = C\y;
% Initial estimates of exponential parameters based on cubic fit.
  b = -2*d(3)/d(2);
  a = -d(2)/b;
  c = d(1) - a;

  a00 = [a b c]';

%%
% NLLS estimates of $\alpha$ using Gauss Newton algorithm
% Function and gradient of exponential decay model
  fun1 = @(a)ED(a,t,0,y,ones(m,1));

% Nonlinear least squares estimate of exponential decay parameters
  ahat = lsqnonlin(fun1,a00);
  n = length(ahat);

% Least squares estimate of sigma based on least squares estimate
  [f,J] = fun1(ahat);
  sighat = norm(f)/sqrt(m-n);

%%
% Prior information

%%
% Prior estimate of $1/\sqrt(\phi)$
  m0 = 10;
  s00 = 1;

%%
% Function to be minimised along with its gradient
  fun = @(a)ED(a,t,0,y,ones(m,1)*sighat);

%%
% Quantiles required from the MCMC samples
  Q = [2.5;50;97.5];

%%
% Sampling from the posterior of $\alpha$ and $\log(\phi)$ using the
% Gaussian random walk algorithm (GRW)
  [S1,aP1,Rh1,Ne1,AA1,IAA1,sc1] = nllsmh_grw(ahat,fun,m0,s00,M,N,M0,Q,false);

%%
% Sampling from the posterior of $\alpha$ and $\phi$ using the
% Normal gamma independence chain algorithm (NGIC)
  [S2,aP2,Rh2,Ne2,AA2,IAA2] = nllsmh_ngic(ahat,fun,m0,s00,1.6,M,N,M0,Q);

%%
% Sampling from the marginal posterior of $\alpha$ using the
% t random walk algorithm (TRW)
  [S3,aP3,Rh3,Ne3,AA3,IAA3,sc3] = nllsmh_trw(ahat,fun,m0,s00,M,N,M0,Q,false);

%%
% Sampling from the marginal posterior of $\alpha$ using the
% t independence chain algorithm (TIC)
  [S4,aP4,Rh4,Ne4,AA4,IAA4] = nllsmh_tic(ahat,fun,m0,s00,1.6,M,N,M0,Q);

%%
% Display convergence indices
  fprintf('Convergence indices:GRW \n')
  fprintf('Parameter %1g: %12.6f \n',[(1:4)',Rh1]')
  fprintf('\n')

  fprintf('Convergence indices:NGIC \n')
  fprintf('Parameter %1g: %12.6f \n',[(1:4)',Rh2]')
  fprintf('\n')

  fprintf('Convergence indices:TRW \n')
  fprintf('Parameter %1g: %12.6f \n',[(1:3)',Rh3]')
  fprintf('\n')

  fprintf('Convergence indices:TIC \n')
  fprintf('Parameter %1g: %12.6f \n',[(1:3)',Rh4]')
  fprintf('\n')

%%
% Display summary statistics
  fprintf('Summary information for posterior distribution: NGIC \n')
  fprintf('\n')
  fprintf('Mean \n');
  for i = 1:4
      fprintf('Parameter %1g:  %12g \n',[i,S2(1,i)])
  end
  fprintf('\n')

  fprintf('Standard deviation \n');
  for i = 1:4
      fprintf('Parameter %1g:  %12g \n',[i,S2(2,i)])
  end
  fprintf('\n')

  fprintf('Median \n');
  for i = 1:4
      fprintf('Parameter %1g:  %12g \n',[i,S2(4,i)])
  end
  fprintf('\n')

  fprintf('2.5 percentile \n');
  for i = 1:4
      fprintf('Parameter %1g:  %12g \n',[i,S2(3,i)])
  end
  fprintf('\n')

  fprintf('97.5 percentile \n');
  for i = 1:4
      fprintf('Parameter %1g:  %12g \n',[i,S2(5,i)])
  end
  fprintf('\n')

%%
% Generate histogram of samples

%%
% Setting the number of bins
  nb = 30;

%%
% Histogram of model parameters
  Rhata = zeros(n,1);
  for i=1:n
    % Samples of $\alpha_i$ from the GRW algorithm
      A1 = AA1(:,:,i);
      A1 = A1(:);
    
    % Samples of $\alpha_i$ from the NGIC algorithm
      A2 = AA2(:,:,i);
      A2 = A2(:);
    
    % Samples of $\alpha_i$ from the TRW algorithm
      A3 = AA3(:,:,i);
      A3 = A3(:);
    
    % Samples of $\alpha_i$ from the TIC algorithm
      A4 = AA4(:,:,i);
      A4 = A4(:);
    
    % Histogram of samples
      figure

      w = (max(A1)-min(A1))/(2*nb);
      wa = linspace(1, 2*nb, 2*nb);
      cen = wa(1:2:end)*w+min(A1);
      edge = [min(A1), wa(2:2:end)*w+min(A1)];

      c1 = histcounts(A1, edge);
      c2 = histcounts(A2, edge);
      c3 = histcounts(A3, edge);
      c4 = histcounts(A4, edge);

      c = [c1', c2', c3', c4'];
      mid = cen';
      
      del = mid(2) - mid(1);
      prop = c./(ones(nb,1)*sum(c)*del);
      bar(mid,prop)
      
    % Axis labels and legend 
      if i == 1
          xlabel('\alpha_1/1')
          ylabel('p(\alpha_1)/1')
          title('MCMC samples: \alpha_1')
      elseif i == 2
          xlabel('\alpha_2/1')
          ylabel('p(\alpha_2)/1')
          title('MCMC samples: \alpha_2')
      elseif i == 3
          xlabel('\alpha_3/1')
          ylabel('p(\alpha_3)/1')
          title('MCMC samples: \alpha_3')
      end
      legend('GRW','NGIC','TRW','TIC')
      
    % Rhat of samples of $\alpha_i$ obtained using different 
    % jumping distributions 
      Rhata(i) = mcmcci([A1 A2 A3 A4],M0*N);
    
  end
%%
% Histogram of precision parameters

% Samples of $\phi$ from the GRW algorithm
  P1 = AA1(:,:,end);
  P1 = exp(P1(:));

% Samples of $\phi$ from the GRW algorithm
  P2 = AA2(:,:,end);
  P2 = P2(:);

% Histogram of samples
  figure

  w = (max(P2)-min(P2))/(2*nb);
  wa = linspace(1, 2*nb, 2*nb);
  cen = wa(1:2:end)*w+min(P2);
  edge = [min(P2), wa(2:2:end)*w+min(P2)];

  c1 = histcounts(P1, edge);
  c2 = histcounts(P2, edge);

  c = [c1', c2'];
  mid = cen';

  del = mid(2) - mid(1);
  prop = c./(ones(nb,1)*sum(c)*del);
  bar(mid,prop)

% Axis labels and legend  
  xlabel('\phi/1')
  ylabel('p(\phi)/1')
  title('MCMC samples: \phi')
  legend('GRW','NGIC')
  
% Rhat of samples of $\phi$ obtained using different jumping distributions 
  Rhatp = mcmcci([P1 P2],M0*N);

%% 
% Display $\hat{R}_a$ and $\hat{R}_{\phi}$

% Rhata 
  fprintf('Rhat alpha \n')
  fprintf('Parameter %1g: %12.6f \n',[(1:3)',Rhata]')
  fprintf('\n')
  
% Rhatp  
  fprintf('Rhat phi: %12.6f \n',Rhatp)
  fprintf('\n')
%% References
% Bayesian Data Analyis, A Gelman, J B Carlin, H S Stern and D B Rubin,
% Chapman & Hall/CRC, Boca Raton, 2004 (Section 11.6)

%% Acknowledgements
% 
% The work described here was supported by the National Measurement System 
% Directorate of the UK Department for Science, Innovation and Technology.

%% Program identifier
%
% NLLSMH Open-Source Software (Release 1.1):
% Software for measurement uncertainty evaluation
%
% A B Forbes and K Jagan, Data Science department,
% National Physical Laboratory, UK
%
% Copyright NPL Management Limited 2025