function [as,l0s] = jump_ngic(ahat,nu,s2bar,R,N)
% -------------------------------------------------------------------------
% jump_ngic.m           Jumping distribution for the Metropolis Hastings
%                       independence chain algorithm with a normal gamma
%                       jumping distribution.
%                       To be used as function handle within mcmcmh.m.
%                       @()jumpngic(ahat,nu,s2bar,R,N).
% -------------------------------------------------------------------------
% K Jagan, A B Forbes, NPL, www.npl.co.uk
% -------------------------------------------------------------------------
% Inputs:
% ahat(n,1)             Non-leaner least squares estimate of alpha.
% nu                    m + m0 - n.
% s2bar                 Weighted mean of prior variance and least squares
%                       estimate of variance of the data.
% R                     Upper traingular matrix obtained by QR
%                       factorisation of the Jacobian matrix evaluated at 
%                       the non-linear least squares estimate.   
% N                     Number of parallel chains.
% Outputs:
% as(p,N)               Proposed parameter array which is randomly sampled
%                       from the jumping distribution. p = n+1.
% l0s(1,N)              Log of the jumping distribution evaluated at as.
% -------------------------------------------------------------------------
% [as,l0s] = jump_ngic(ahat,nu,s2bar,R,N)
% -------------------------------------------------------------------------
% Size of the data
  n = length(ahat);

% Parameters of Gamma jumping distribution
  k1 = nu/2;
  k2 = nu*s2bar/2;

% Proposal samples from phi|y
  Pp = gamrnd(k1,1/k2,1,N);
  Ss = ones(1,N)./sqrt(Pp);
%
% Proposal samples from alpha|phi,y
  E = randn(n,N);
  Aa = ahat*ones(1,N) + repmat(Ss,n,1).*(R\E);

% Proposal sample
  as = [Aa;Pp];

% Evaluating log of the jumping density at as
  k3 = (nu + n)/2 - 1;
  l0s = k3*log(Pp) - Pp/2.*(nu*s2bar) - sum(E.*E)/2;
% -------------------------------------------------------------------------
% End of jump_ngic.m
% -------------------------------------------------------------------------
end
