function T = tar(ad, y, s0, m0d, s0d, m0a)
% -------------------------------------------------------------------------
% tar.m             The log of the posterior distribution of $\log(\alpha)$
%                   and $\log(delta)$ for the Bayesian problem.
%
%                   $y_i \sim N(\alpha \delta, \sigma_0^2)$
%                   $\alpha \sim G(m_{0,a}/2, m_{0,a}/2)$
%                   $\delta \sim G(m_{0,d}/2,m_{0,d}\sigma_{0,d}^2/2)$
% -------------------------------------------------------------------------
% K Jagan, A B Forbes, NPL, www.npl.co.uk
% -------------------------------------------------------------------------
% Input parameters
% ad(2,N)           Values of log alpha and log delta for N chains.
% y(m,1)            Data vector.
% s0                The known standard deviation of y.
% m0d               Parameter of the prior for delta.
% s0d               Parameter of the prior for delta.
% m0a               Parameter of the prior for alpha.
% Output parameter
% T(1,N)            Log of the posterior distribution of log alpha and log
%                   phi.
% -------------------------------------------------------------------------
% T = tar(ad, y, s0, m0d, s0d, m0a)
% -------------------------------------------------------------------------

% number of parallel chains
  N = size(ad, 2);

% length of data vector
  m = length(y);

% extracting alpha and delta
  A = exp(ad(1, :));
  D = exp(ad(2, :));

% likelihood based on Gaussian model
  lik = -sum((y*ones(1, N) - repmat(A.*D, m, 1)).^2)/(2*s0^2);

% Gamma prior for alpha
  pria = m0a/2*log(A) - A*m0a/2;

% Gamma prior for delta
  prid = m0d/2*log(D) - D*m0d*s0d^2/2;

% posterior for log alpha and log delta
  T = lik + pria + prid;

% -------------------------------------------------------------------------
% End of tar.m
% -------------------------------------------------------------------------
end