function [abar,s,aQ,V,AA] = mcmcsum(A,M0,Q)
% -------------------------------------------------------------------------
% MCMCSUM.M        Summary information from MC samples.
% -------------------------------------------------------------------------
% A B Forbes, K Jagan, I M Smith, NPL, www.npl.co.uk
% -------------------------------------------------------------------------
% Design documentation 
% [1] A B Forbes, An MCMC algorithm based on GUM Supplement 1 for 
% uncertainty evaluation, Measurement, 2012, 
% DOI:10.1016/j.measurement.2012.01.018
%
% [2] A B Forbes et al, User manual for MCM2MCMC: Software implementing a
% Markov chain Monte Carlo algorithm for uncertainty evaluation based on 
% Supplement 1 to the GUM, NPL Report MS/20, 2018
% -------------------------------------------------------------------------
% Input parameters
% A(M,N,L)         An array that stores samples of size M x N
%                  for each of L parameters.
% M0               Burn-in period with M > M0 >= 0.
% Q(nQ,1)          Percentiles specifications, 0 <= Q(i) <= 100.
%
% Output parameters
% abar(L,1)        Mean for each sample.
% s(L,1)           Standard deviation for sample.
% aQ(nQ,L)         Percentiles corresponding to Q.
%                  If Q(i) =   0, aQ(i,j) = min(A{j});
%                     Q(i) = 100, aQ(i,j) = max(A{j}).
% V(L,L)           Sample covariance matrix.
% AA(n,L)          Reshaped samples, n = (M-M0)*N.
%                  Note: If Aj = A(:,:,j) then abar, s, aQ and V are 
%                  calculated from reshape(Aj(M0+1:M,:),(M-M0)*N,1); 
% -------------------------------------------------------------------------
% [abar,s,aQ,V,AA] = mcmcsum(A,M0,Q)
% -------------------------------------------------------------------------
  L = size(A,3);
  [M,N] = size(A(:,:,1));
  m = (M-M0)*N;
% 
% Initialisation.
  abar = zeros(L,1);
  s = zeros(L,1);
  if nargout > 2
      nQ = length(Q);
      aQ = zeros(nQ,L);
  end % if nargout 
  if nargout > 3
      V = zeros(L,L);
      AA = zeros(m,L);
  end % if nargout 
%
% Calculate summary statistics for each parameter. 
  for j = 1:L
    Aj = A(:,:,j);
    aaj = Aj(M0+1:M,:);
    aaj = aaj(:);
%   Calculate mean and standard deviation. 
    abar(j) = mean(aaj);
    s(j) = std(aaj);
%   Calculate quantiles. 
    if nargout > 2
      aajs = sort(aaj);
      for q = 1:nQ
        iq = round(Q(q)*m/100);
        iq = max([iq;1]);
        aQ(q,j) = aajs(iq);
      end % for q 
    end % if nargout 
%   Reshape sample. 
    if nargout > 3
      AA(:,j) = aaj;
    end % if nargout 
  end % for j 
%   
% Calculate sample covariance matrix.
  if nargout > 3
    V = cov(AA);
  end
% -------------------------------------------------------------------------
% End of mcmcsum.m
% -------------------------------------------------------------------------